<?php

use Illuminate\Support\Facades\Route;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\Status;
use App\Models\Task;
use App\Models\CommunicationBoard;
use App\Http\Controllers\TaskController;
use App\Http\Controllers\UserController;

// ステータスとタスク一覧
Route::get('/tasks', [TaskController::class, 'selectTaskList']);

// ステータスとタスクをステータスごとにグループ化して取得
Route::get('/statuses-grouped', [TaskController::class, 'selectStatusTaskList']);

// ステータス名変更
Route::post('/status/{id}', function (Request $request, $id) {
    $status = Status::find($id);

    if (!$status) {
        return response()->json(['message' => 'ステータスが見つかりません'], 404);
    }
    // ステータス名が存在すれば更新
    if ($request->filled('status_name')) {
        $status->status_name = $request->input('status_name');
        $status->save();
    }

    return response()->json(['message' => '更新成功']);
});

// タスクの更新
Route::put('/tasks/{id}', function (Request $request, $id) {
    $task = Task::find($id);
    if (!$task) {
        return response()->json(['message' => 'タスクが見つかりません'], 404);
    }
    // 更新可能なフィールド一覧
    $updatableFields = ['status_id', 'task_title', 'start_date', 'end_date', 'priority_id', 'worker', 'reviewer', 'label', 'comment'];
    // 各フィールドについて、存在すれば更新
    $task->fill($request->only($updatableFields))->save();
    return response()->json(['message' => '更新成功']);
});

// ステータス一覧取得
Route::get('/status/{id}', function () {
    $statuses = Status::where('pj_id', 1)->get(['status_id', 'status_name']);
    if ($statuses->isEmpty()) {
        return response()->json(['message' => 'ステータスが見つかりません'], 404);
    }
    return response()->json($statuses);
});

// ユーザーデータ取得
Route::get('/users', [UserController::class, 'selectUsers']);

// 優先度マスタデータ取得
Route::get('/priority', function () {
    $priorities = DB::table('mst_priorities')->get(['priority_id', 'priority_name']);
    if ($priorities->isEmpty()) {
        return response()->json(['message' => '優先度が見つかりません'], 404);
    }
    return response()->json($priorities);
});

// ステータス追加
Route::put('/addstatus/{id}', function (Request $request, $id) {
    $status = new Status();
    $status->status_name = $request->input('status_name');
    $status->pj_id = $request->input('pj_id');
    $status->save();
    return response()->json([
        'status_id' => $status->status_id,
        'status_name' => $status->status_name
    ]);
});

// ステータス削除
Route::delete('/statuses/{id}', function ($id) {
    $status = Status::find($id);
    if (!$status) {
        return response()->json(['message' => 'ステータスが見つかりません'], 404);
    }
    $status->delete();
    return response()->json(['message' => '削除成功']);
});

// タスク追加
Route::post('/tasks', function (Request $request) {
    $task = new Task();
    $task->task_title = $request->input('task_title');
    $task->status_id = $request->input('status_id');
    $task->start_date = $request->input('start_date');
    $task->end_date = $request->input('end_date');
    $task->priority_id = $request->input('priority_id');
    $task->worker = $request->input('worker');
    $task->reviewer = $request->input('reviewer');
    $task->label = $request->input('label');
    $task->comment = $request->input('comment');
    $task->save();
    return response()->json([
        'task_id' => $task->task_id,
        'task_title' => $request->input('task_title'),
        'status_id' => $request->input('status_id'),
        'start_date' => $request->input('start_date'),
        'end_date' => $request->input('end_date'),
        'priority_id' => $request->input('priority_id'),
        'worker' => $request->input('worker'),
        'reviewer' => $request->input('reviewer'),
        'label' => $request->input('label'),
        'comment' => $request->input('comment')
    ])->setStatusCode(201, 'タスクが作成されました');
});     

// タスクコピー
Route::post('/tasks/copy', function (Request $request) {
    $taskIds = $request->input('task_ids');
    if (!is_array($taskIds)) {
        return response()->json(['error' => '不正なデータ形式です'], 422);
    }
    $request->validate([
        'task_ids' => 'required|array',
        'task_ids.*' => 'integer',
    ]);
    try {
        $tasks = Task::whereIn('task_id', $taskIds)->get();
        foreach ($tasks as $task) {
            $newTask = $task->replicate();
            $newTask->save();
        }
        return response()->json(['message' => '削除成功']);
    } catch (\Exception $e) {
        return response()->json(['error' => '削除に失敗しました', 'detail' => $e->getMessage()], 500);
    }
});

// タスク削除
Route::delete('/tasks/delete', function (Request $request) {
    $taskIds = $request->input('task_ids');
    if (!is_array($taskIds)) {
        return response()->json(['error' => '不正なデータ形式です'], 422);
    }
    $request->validate([
        'task_ids' => 'required|array',
        'task_ids.*' => 'integer',
    ]);
    try {
        Task::whereIn('task_id', $taskIds)->delete();
        return response()->json(['message' => '削除成功']);
    } catch (\Exception $e) {
        return response()->json(['error' => '削除に失敗しました', 'detail' => $e->getMessage()], 500);
    }
});

Route::delete('/tasks/{id}', function ($id) {
    $task = Task::find($id);
    if (!$task) {
        return response()->json(['message' => 'タスクが見つかりません'], 404);
    }
    $task->delete();
    return response()->json(['message' => '削除成功']);
});


Route::get('/users_tasks', function () {
    $results = DB::select("
        SELECT
            * 
        FROM
            ( 
                SELECT
                    task.worker AS user_id
                    , worker.user_name AS user_name
                    , task.task_id
                    , task.task_title
                    , task.status_id
                    , status.status_name
                    , task.end_date
                    , task.worker
                    , worker.user_name AS worker_name
                    , task.reviewer
                    , reviewer.user_name AS reviewer_name
                    , task.priority_id
                    , priority.priority_name 
                FROM
                    tr_tasks task 
                    LEFT OUTER JOIN mst_users reviewer 
                        ON task.reviewer = reviewer.user_id 
                    LEFT OUTER JOIN mst_users worker 
                        ON task.worker = worker.user_id 
                    LEFT OUTER JOIN tr_statuses status 
                        ON status.status_id = task.status_id 
                    LEFT OUTER JOIN mst_priorities priority 
                        ON priority.priority_id = task.priority_id 
                UNION 
                SELECT
                    task.reviewer AS id
                    , reviewer.user_name AS name
                    , task.task_id
                    , task.task_title
                    , task.status_id
                    , status.status_name
                    , task.end_date
                    , task.worker
                    , worker.user_name AS worker_name
                    , task.reviewer
                    , reviewer.user_name AS reviewer_name
                    , task.priority_id
                    , priority.priority_name 
                FROM
                    tr_tasks task 
                    LEFT OUTER JOIN mst_users reviewer 
                        ON task.reviewer = reviewer.user_id 
                    LEFT OUTER JOIN mst_users worker 
                        ON task.worker = worker.user_id 
                    LEFT OUTER JOIN tr_statuses status 
                        ON status.status_id = task.status_id 
                    LEFT OUTER JOIN mst_priorities priority 
                        ON priority.priority_id = task.priority_id
            ) users_tasks 
        WHERE
            users_tasks.user_id IS NOT NULL 
        ORDER BY
            users_tasks.user_id, users_tasks.end_date
    ");

    // コレクションに変換
    $collection = collect($results);

    // user_idごとにグループ化
    $grouped = $collection->groupBy('user_id')->map(function ($tasks, $userId) {
        return [
            'user_id' => $userId,
            'user_name' => $tasks->first()->user_name,
            'count' => $tasks->count(),
            'tasks' => $tasks->values() // タスク配列
        ];
    })->values();

    // レスポンスとして返す
    return response()->json($grouped);
});

// 連絡ボードデータ取得
Route::get('/communicationboards', function () {
    $tweets = DB::table('tr_communication_boards')
        ->select(
            'tr_communication_boards.communication_board_id',
            'tr_communication_boards.text_in_board'
        )
        ->where('pj_id', 1)
        ->get();
    return response()->json($tweets);
});

// ステータス名変更
Route::put('/communicationboard/{id}', function (Request $request, $id) {
    $cb = CommunicationBoard::find($id);

    if (!$cb) {
        return response()->json(['message' => '連絡ボードが見つかりません'], 404);
    }
    $cb->text_in_board = $request->input('text_in_board');
    $cb->save();

    return response()->json(['message' => '更新成功']);
});